/**
 * Format price in triple currency format: BIF | USD | EUR
 * @param {number|string} bif - Burundian Franc price
 * @param {number|string} usd - US Dollar price
 * @param {number|string} eur - Euro price
 * @returns {string} Formatted price string like "1200000 BIF | 60 USD | 55 EUR"
 */
export const formatPriceTriple = (bif, usd, eur) => {
    const parts = [];
    
    if (bif !== undefined && bif !== null && bif !== '') {
        const bifFormatted = `${parseInt(bif).toLocaleString()} BIF`;
        parts.push(bifFormatted);
    }

    if (usd !== undefined && usd !== null && usd !== '') {
        const usdFormatted = `${parseFloat(usd).toFixed(0)} USD`;
        parts.push(usdFormatted);
    }

    if (eur !== undefined && eur !== null && eur !== '') {
        const eurFormatted = `${parseFloat(eur).toFixed(0)} EUR`;
        parts.push(eurFormatted);
    }

    return parts.join(' | ');
};

/**
 * Format price triple currency - returns object with individual formatted prices
 * @param {number|string} bif - Burundian Franc price
 * @param {number|string} usd - US Dollar price
 * @param {number|string} eur - Euro price
 * @returns {Object} Object with bifPrice, usdPrice, eurPrice properties
 */
export const formatPriceTripleObject = (bif, usd, eur) => {
    return {
        bifPrice: bif ? `${parseInt(bif).toLocaleString()} BIF` : null,
        usdPrice: usd ? `${parseFloat(usd).toFixed(0)} USD` : null,
        eurPrice: eur ? `${parseFloat(eur).toFixed(0)} EUR` : null,
        combined: formatPriceTriple(bif, usd, eur)
    };
};

/**
 * Format price in compact triple currency format for list views
 * Uses shorter format: "1200000 BIF | 60 USD | 55 EUR"
 * @param {number|string} bif - Burundian Franc price
 * @param {number|string} usd - US Dollar price
 * @param {number|string} eur - Euro price
 * @returns {string} Compact formatted price string
 */
export const formatPriceTripleCompact = (bif, usd, eur) => {
    return formatPriceTriple(bif, usd, eur); // Same format, but kept for semantic clarity
};

