import { Link, usePage } from '@inertiajs/react';
import { formatPriceTriple } from '@/utils/priceFormatter';

export default function Event({ eventRooms = [] }) {
    const { siteinfo } = usePage().props;
    
    // Get dynamic description
    const eventsDescription = siteinfo?.events_description || '';
    // Generate event room items - use event rooms if available, otherwise use 3 placeholders
    const getEventRoomItems = () => {
        if (eventRooms && eventRooms.length > 0) {
            return eventRooms.map((room) => ({
                id: room.id != null ? String(room.id) : '',
                name: room.name,
                image: room.primary_photo ? `/storage/${room.primary_photo}` : 'https://placehold.co/800x571',
                price: formatPriceTriple(room.price_fbu, room.price_usd, room.price_eur) || 'Price on request',
                capacity: room.capacity,
                layouts: room.layouts && Array.isArray(room.layouts) ? room.layouts.join(', ') : '',
                shortDescription: room.short_description || '',
            }));
        }

        // Return 3 placeholder items if no event rooms
        return Array.from({ length: 3 }, (_, index) => ({
            id: `placeholder-${index + 1}`,
            name: `Event Room ${index + 1}`,
            image: 'https://placehold.co/800x571',
            price: '$0.00',
            capacity: null,
            layouts: '',
            shortDescription: 'Far far away, behind the word mountains, far from the countries Vokalia and Consonantia, there live the blind texts.',
        }));
    };

    const eventRoomItems = getEventRoomItems();

    return(
        <>
        <section className="section">
        <div className="container">
            {(eventsDescription || eventRoomItems.length > 0) && (
                <div className="row justify-content-center text-center mb-5">
                    <div className="col-md-7">
                        <h2 className="heading" data-aos="fade-up">Meeting &amp; Event Rooms</h2>
                        {eventsDescription && (
                            <p data-aos="fade-up" data-aos-delay="100">{eventsDescription}</p>
                        )}
                    </div>
                </div>
            )}
            <div className="row">
            {eventRoomItems.map((room, index) => (
                <div key={room.id} className="col-md-6 col-lg-4" data-aos="fade-up" data-aos-delay={index * 100}>
                    <Link href={room.id && String(room.id).startsWith('placeholder') ? '#' : `/meetings-events/${room.id}`} className="room">
                    <figure className="img-wrap">
                        <img src={room.image} alt={room.name} className="img-fluid mb-3" />
                    </figure>
                    <div className="p-3 text-center room-info">
                        <h2>{room.name}</h2>
                        {room.capacity && (
                            <p className="mb-1 text-muted"><small>Capacity: {room.capacity} people</small></p>
                        )}
                        <span className="text-uppercase letter-spacing-1">{room.price} / per day</span>
                        {room.layouts && (
                            <p className="mt-2 mb-0"><small className="text-muted">{room.layouts}</small></p>
                        )}
                    </div>
                    </Link>
                </div>
            ))}
            </div>
        </div>
        </section>
        </>
    );
}