import Card from "@admin/ui/Card";
import Col from "@admin/ui/Col";
import FormGroup from "@admin/ui/forms/FormGroup";
import Switch from "@admin/ui/forms/Switch";
import Row from "@admin/ui/Row";
import { useForm } from "@inertiajs/react";
import { useFormResponse } from '@/hooks/useFormResponse';
import { Toaster } from "react-hot-toast";

export default function Room({ accommodations = [] }) {
    const { data, setData, post, processing, errors, reset } = useForm({
        'room_number': '',
        'accommodation_id': '',
        'is_available': true,
    });

    const handleSubmit = (event) => {
        event.preventDefault();
        
        const formResponse = useFormResponse(reset);
        post('/admin/rooms/store', {
            ...formResponse,
            preserveScroll: true,
        });
    }

    return (
        <>
            <Toaster />
            <Row md="12">
                <Col lg="12">
                    <h2 className="page-title">Add New Room</h2>
                    <p className="text-muted">Create a new individual room instance</p>
                    <Row>
                        <Col lg="12">
                            <Card md="12" title="Room Details">
                                <form onSubmit={handleSubmit}>
                                    <Row>
                                        <Col md="6">
                                            <FormGroup
                                                id="room_number"
                                                label="Room Number"
                                                type="text"
                                                value={data.room_number}
                                                onChange={(e) => setData("room_number", e.target.value)}
                                                required
                                                invalidFeedback={errors.room_number}
                                                helpText="Enter the room number (e.g., 101, 102, 201)"
                                            />

                                            <FormGroup
                                                id="accommodation_id"
                                                label="Room Category"
                                                as="select"
                                                value={data.accommodation_id}
                                                onChange={(e) => setData("accommodation_id", e.target.value)}
                                                required
                                                invalidFeedback={errors.accommodation_id}
                                                helpText="Select the accommodation category this room belongs to"
                                            >
                                                <option value="">Select Category</option>
                                                {accommodations.map((accommodation) => (
                                                    <option key={accommodation.id} value={accommodation.id}>
                                                        {accommodation.name}
                                                    </option>
                                                ))}
                                            </FormGroup>

                                            <div className="form-group">
                                                <Switch
                                                    id="is_available"
                                                    label="Room Available"
                                                    checked={data.is_available}
                                                    onChange={(checked) => setData('is_available', checked)}
                                                    helpText="Toggle room availability status"
                                                />
                                            </div>

                                            <div className="form-group">
                                                <button 
                                                    type="submit" 
                                                    className="btn btn-primary"
                                                    disabled={processing}
                                                >
                                                    {processing ? 'Creating...' : 'Create Room'}
                                                </button>
                                                <a 
                                                    href="/admin/rooms" 
                                                    className="btn btn-secondary ml-2"
                                                >
                                                    Cancel
                                                </a>
                                            </div>
                                        </Col>
                                    </Row>
                                </form>
                            </Card>
                        </Col>
                    </Row>
                </Col>
            </Row>
        </>
    );
}



