import FormGroup from "@admin/ui/forms/FormGroup";
import Switch from "@admin/ui/forms/Switch";
import FileUploadInput from "@admin/ui/forms/FileUploadInput";
import MultiImageUploadInput from "@admin/ui/forms/MultiImageUploadInput";
import Modal from "@admin/ui/Modal";
import Row from "@admin/ui/Row";
import Col from "@admin/ui/Col";
import { useEffect, useState } from "react";
import { useFormResponse } from '@/hooks/useFormResponse';
import { useForm } from "@inertiajs/react";
import { toast } from "react-hot-toast";

export default function EditAccommodation({ isOpen, onClose, initialData }){
    const {data, setData, put, processing, errors} = useForm({
        'room_name': '',
        'room_category': '',
        'primary_photo': '',
        'price_fbu': '',
        'price_usd': '',
        'price_eur': '',
        'description': '',
        'full_description': '',
        'size': '',
        'bed_type': '',
        'view': '',
        'amenities': [],
        'gallery': [],
        'status': true,
        'featured': false,
    });

    const [presetPreviews, setPresetPreviews] = useState([]);
    // Store raw text for amenities to allow typing commas
    const [amenitiesText, setAmenitiesText] = useState('');

    useEffect(() => {
        if (initialData) {
          let gallery = [];
          if (initialData?.gallery) {
              if (Array.isArray(initialData.gallery)) {
                  gallery = initialData.gallery;
              } else if (typeof initialData.gallery === 'string') {
                  try {
                      const parsed = JSON.parse(initialData.gallery);
                      gallery = Array.isArray(parsed) ? parsed : [];
                  } catch (e) {
                      gallery = [];
                  }
              }
          }
          
          const validGallery = gallery.filter(item => item && typeof item === 'string' && item.trim() !== '');
          
          const amenities = initialData?.amenities || [];
          setData({
              room_name: initialData?.room_name || '',
              room_category: initialData?.room_category || '',
              primary_photo: initialData?.primary_photo || '',
              price_fbu: initialData?.price_fbu || '',
              price_usd: initialData?.price_usd || '',
              price_eur: initialData?.price_eur || '',
              description: initialData?.description || '',
              full_description: initialData?.full_description || '',
              size: initialData?.size || '',
              bed_type: initialData?.bed_type || '',
              view: initialData?.view || '',
              amenities: amenities,
              gallery: validGallery,
              status: initialData?.status === 1 || initialData?.status === true,
              featured: initialData?.featured === 1 || initialData?.featured === true,
          });

          setAmenitiesText(Array.isArray(amenities) ? amenities.join(', ') : '');

          if (validGallery.length > 0) {
              setPresetPreviews(validGallery.map(filename => `/storage/${filename}`));
          } else {
              setPresetPreviews([]);
          }
        } else {
            setPresetPreviews([]);
        }
    }, [initialData, setData]);

    const handlePrimaryPhotoSuccess = (filename) => {
        if (filename) {
            setData('primary_photo', filename);
        }
    };

    const handleGallerySuccess = ({ filenames }) => {
        const validFilenames = Array.isArray(filenames) ? filenames.filter(f => f && f.trim() !== '') : [];
        setData('gallery', validFilenames);
        setPresetPreviews(validFilenames.map(filename => `/storage/${filename}`));
    };

    const handleError = (error) => {
        toast.error('Failed to upload image');
    };

    const handleAmenitiesChange = (e) => {
        const value = e.target.value;
        setAmenitiesText(value);
        const amenitiesArray = value.split(',')
            .map(item => item.trim())
            .filter(item => item.length > 0);
        setData('amenities', amenitiesArray);
    };

    const handleAmenitiesBlur = () => {
        setAmenitiesText(data.amenities?.join(', ') || '');
    };

    const handleSave = (event) => {
        event.preventDefault();
        const formResponse = useFormResponse(() => {
            onClose?.(); 
        });
    
        put(`/admin/api/update_accommodation/${initialData.id}`, {
            preserveScroll: true,
            onSuccess: formResponse.onSuccess, 
            onError: formResponse.onError, 
        });
    };    

    return(
    <>
    <Modal isOpen={isOpen} onClose={onClose} title={`Edit ${initialData?.room_name || 'Accommodation'}`} size="large"
    primaryButtonText="Save Changes"
    onPrimaryAction={handleSave}
    processing={processing}
    >
        <form onSubmit={handleSave}>
            <Row>
                <Col md="6">
                    <FormGroup
                        id="room_category"
                        label="Room Category"
                        type="text"
                        value={data.room_category}
                        onChange={(e) => {
                            setData("room_category", e.target.value);
                            // Auto-populate room_name from category if room_name is empty or matches old category
                            if (!data.room_name || data.room_name === initialData?.room_category) {
                                setData("room_name", e.target.value);
                            }
                        }}
                        required
                        invalidFeedback={errors.room_category}
                        helpText="Enter the room category (e.g., Simple, Standard, Silver, Golden, VIP). This will also be used as the room name."
                    />

                    <FormGroup 
                        label="Price (BIF)" 
                        type="number"
                        value={data.price_fbu}
                        onChange={(e) => setData('price_fbu', e.target.value)}
                        helpText="Enter price in Burundian Franc"
                        invalidFeedback={errors.price_fbu}
                        min="0.01"
                        step="0.01"
                        required
                    />

                    <FormGroup 
                        label="Price (USD)" 
                        type="number"
                        value={data.price_usd}
                        onChange={(e) => setData('price_usd', e.target.value)}
                        helpText="Enter price in US Dollars"
                        invalidFeedback={errors.price_usd}
                        min="0.01"
                        step="0.01"
                        required
                    />

                    <FormGroup 
                        label="Price (EUR)" 
                        type="number"
                        value={data.price_eur}
                        onChange={(e) => setData('price_eur', e.target.value)}
                        helpText="Enter price in Euros"
                        invalidFeedback={errors.price_eur}
                        min="0.01"
                        step="0.01"
                    />

                    <FormGroup
                        id="size"
                        label="Room Size"
                        type="text"
                        value={data.size}
                        onChange={(e) => setData("size", e.target.value)}
                        helpText="e.g., 25 sqm"
                        invalidFeedback={errors.size}
                    />

                    <FormGroup
                        id="bed_type"
                        label="Bed Type"
                        type="text"
                        value={data.bed_type}
                        onChange={(e) => setData("bed_type", e.target.value)}
                        helpText="e.g., King Bed, Twin Beds"
                        invalidFeedback={errors.bed_type}
                    />

                    <FormGroup
                        id="view"
                        label="View"
                        type="text"
                        value={data.view}
                        onChange={(e) => setData("view", e.target.value)}
                        helpText="e.g., Ocean View, Garden View"
                        invalidFeedback={errors.view}
                    />
                </Col>
                <Col md="6">
                    <FormGroup
                        id="description"
                        label="Brief Description"
                        as="textarea"
                        value={data.description}
                        helpText="Provide a brief description (minimum 20 characters)"
                        onChange={(e) => setData("description", e.target.value)}
                        rows="4"
                        invalidFeedback={errors.description}
                        required
                    />

                    <FormGroup
                        id="full_description"
                        label="Full Description"
                        as="textarea"
                        value={data.full_description}
                        helpText="Detailed description for the room"
                        onChange={(e) => setData("full_description", e.target.value)}
                        rows="5"
                        invalidFeedback={errors.full_description}
                    />

                    <FormGroup
                        id="amenities"
                        label="Amenities"
                        as="textarea"
                        value={amenitiesText}
                        helpText="Enter amenities separated by commas"
                        onChange={handleAmenitiesChange}
                        onBlur={handleAmenitiesBlur}
                        rows="3"
                        invalidFeedback={errors.amenities}
                    />
                    {data.amenities.length > 0 && (
                        <div className="mb-2">
                            <small className="text-muted">Current amenities: </small>
                            {data.amenities.map((amenity, index) => (
                                <span key={index} className="badge badge-secondary mr-1">{amenity}</span>
                            ))}
                        </div>
                    )}
                </Col>
            </Row>

            <Row>
                <Col md="6">
                    <FileUploadInput 
                        label="Upload Primary Photo (800x571)px" 
                        buttonText={"Choose file (800x571)"}
                        uploadUrl="/admin/api/accommodation/upload_image"
                        onSuccess={handlePrimaryPhotoSuccess}
                        onError={handleError}
                        previewUrl={data.primary_photo ? `/storage/${data.primary_photo}` : ''}
                        invalidFeedback={errors.primary_photo}
                    />
                </Col>
                <Col md="6">
                    <label className="label">Gallery Photos (1123x700)px</label>
                    <MultiImageUploadInput
                        label="Upload Gallery Images (1123x700)px"
                        inputId="galleryUploadEdit"
                        buttonText="Choose gallery images (1123x700)"
                        uploadUrl="/admin/api/accommodation/upload_multiple"
                        onSuccess={handleGallerySuccess}
                        onError={handleError}
                        presetPreviews={presetPreviews}
                    />
                    <small className="text-muted d-block mt-1">Recommended size: 1123x700 pixels</small>
                    {errors.gallery && (
                        <div className="invalid-feedback d-block">{errors.gallery}</div>
                    )}
                </Col>
            </Row>

            <Switch
                id="statusSwitch"
                label="Active"
                checked={data.status}
                onChange={({ target: { checked } }) =>
                    setData(prevData => ({ ...prevData, status: checked }))
                }                  
            />
            <Switch
                id="featuredSwitch"
                label="Featured"
                checked={data.featured}
                onChange={({ target: { checked } }) =>
                    setData(prevData => ({ ...prevData, featured: checked }))
                }                  
            />
        </form>
    </Modal>
    </>);
}

