import React, { useState, useMemo, useEffect } from 'react';

const StyledTable = ({ 
  data = [], 
  columns = [], 
  itemsPerPage = 4,
  showEdit = true,
  showConfirm = false, 
  showRead =false,
  showSearch = false, 
  title = "Styled Table",
  onView = () => {},
  onRead =() => {},
  onEdit = () => {},
  onDelete = () => {},
  onConfirm = () => {},
  onRowClick = () => {},
  noDataText = "No data available",
}) => {
  const [currentPage, setCurrentPage] = useState(1);
  const [searchTerm, setSearchTerm] = useState('');

  // Filter data based on searchTerm
  const filteredData = useMemo(() => {
    if (!searchTerm) return data;
    return data.filter((row) => {
      return Object.values(row).some((val) =>
        String(val).toLowerCase().includes(searchTerm.toLowerCase())
      );
    });
  }, [searchTerm, data]);

  const totalPages = Math.ceil(filteredData.length / itemsPerPage);
  const startIndex = (currentPage - 1) * itemsPerPage;
  const endIndex = startIndex + itemsPerPage;
  const currentData = filteredData.slice(startIndex, endIndex);

  useEffect(() => {
    if (currentData.length === 0 && currentPage > 1) {
      setCurrentPage((prev) => Math.max(prev - 1, 1));
    }
  }, [currentData, currentPage]);

  return (
    <div className="col-12 mt-2">
      <div className="card">
        <div className="card-body">
          <h4 className="header-title">{title}</h4>

          {/* Search input, shown only if showSearch is true */}
          {showSearch && (
            <div className="mb-3">
              <input
                type="text"
                className="form-control"
                placeholder="Search..."
                value={searchTerm}
                onChange={(e) => {
                  setSearchTerm(e.target.value);
                  setCurrentPage(1);
                }}
              />
            </div>
          )}

          <div className="single-table">
            <div className="table-responsive">
            <table className="table datatables" id="dataTable-1">
            <thead>
              <tr>
                <th scope="col">SN</th>
                {columns.map((column, index) => (
                  <th key={index} scope="col">{column.header}</th>
                ))}
                <th scope="col">Action</th>
              </tr>
            </thead>
            <tbody>
              {currentData.length > 0 ? (
                currentData.map((row, index) => (
                  <tr 
                    key={index}
                    style={{ cursor: onRowClick ? 'pointer' : 'default' }}
                    onClick={(e) => {
                      if (!onRowClick || e.target.closest('ul')) {
                        return;
                      }
                      onRowClick(row);
                    }}
                  >
                    <td>{startIndex + index + 1}</td>
                    {columns.map((column, colIndex) => (
                      <td key={colIndex}>
                        {column.key === "photo" ? (
                          <img src={`/storage/${row[column.key]}`} alt="Photo" width="50" height="50" />
                        ) : column.key === "destinations" && Array.isArray(row[column.key]) ? (
                          row[column.key].map(dest => dest.name).join(", ")
                        ) : column.key === "status" ? (
                          <span className={`badge ${row[column.key] === 1 || row[column.key] === true || row[column.key] === "active" || row[column.key] === "confirmed" ? "badge-success" : "badge-danger"}`}>
                            {row[column.key] === 1 || row[column.key] === true || row[column.key] === "active" || row[column.key] === "confirmed" ? "Active" : "Inactive"}
                          </span>
                        ) : column.key==="is_read" ? (
                          <span className={`badge ${row[column.key] === 1  ? "badge-success" : "badge-danger"}`}>
                            {row[column.key] === 1  ? "Yes" : "No"}
                          </span>
                        ):(
                          row[column.key]
                        )}
                      </td>
                    ))}
                    <td>
                      <ul className="d-flex justify-content-start list-unstyled">
                        {showEdit && (
                          <li className="mr-3">
                            <a
                              href="#"
                              className="text-secondary"
                              onClick={(e) => {
                                e.preventDefault();
                                onEdit(row);
                              }}
                            >
                              <i className="fe fe-edit"></i>
                            </a>
                          </li>
                        )}
                        {showConfirm && (
                          <li className="mr-3">
                            <a
                              href="#"
                              className="text-success"
                              onClick={(e) => {
                                e.preventDefault();
                                onConfirm(row);
                              }}
                            >
                              <i className="fe fe-check-circle"></i>
                            </a>
                          </li>
                        )}
                        {showRead && (
                          <li className="mr-3">
                            <a
                              href="#"
                              className="text-success"
                              onClick={(e) => {
                                e.preventDefault();
                                onRead(row);
                              }}
                            >
                            <i className="fe fe-eye"></i> Read
                            </a>
                          </li>
                        )}
                        <li>
                          <a
                            href="#"
                            className="text-danger"
                            onClick={(e) => {
                              e.preventDefault();
                              onDelete(row);
                            }}
                          >
                            <i className="fe fe-trash"></i>
                          </a>
                        </li>
                      </ul>
                    </td>
                  </tr>
                ))
              ) : (
                <tr>
                  <td colSpan={columns.length + 2} className="text-center py-4">
                    <i className="ti-info-alt mr-2"></i>
                    {noDataText}
                  </td>
                </tr>
              )}
            </tbody>
          </table>

            </div>
          </div>

          {/* Pagination remains unchanged */}
          {filteredData.length > 0 && (
            <div className="col-lg-12 mt-3">
              <nav aria-label="Page navigation example">
                <ul className="pagination">
                  <li className={`page-item ${currentPage === 1 ? 'disabled' : ''}`}>
                    <a 
                      className="page-link" 
                      href="#"
                      onClick={(e) => {
                        e.preventDefault();
                        if (currentPage > 1) setCurrentPage(currentPage - 1);
                      }}
                      aria-label="Previous"
                    >
                      <span aria-hidden="true">&laquo;</span>
                      <span className="sr-only">Previous</span>
                    </a>
                  </li>
                  
                  {[...Array(totalPages)].map((_, i) => (
                    <li 
                      key={i} 
                      className={`page-item ${currentPage === i + 1 ? 'active' : ''}`}
                    >
                      <a 
                        className="page-link" 
                        href="#"
                        onClick={(e) => {
                          e.preventDefault();
                          setCurrentPage(i + 1);
                        }}
                      >
                        {i + 1}
                      </a>
                    </li>
                  ))}

                  <li className={`page-item ${currentPage === totalPages ? 'disabled' : ''}`}>
                    <a 
                      className="page-link" 
                      href="#"
                      onClick={(e) => {
                        e.preventDefault();
                        if (currentPage < totalPages) setCurrentPage(currentPage + 1);
                      }}
                      aria-label="Next"
                    >
                      <span aria-hidden="true">&raquo;</span>
                      <span className="sr-only">Next</span>
                    </a>
                  </li>
                </ul>
              </nav>
            </div>
          )}
        </div>
      </div>
    </div>
  );
};

export default StyledTable;
