<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Accommodation extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'room_name',
        'room_number',
        'room_category',
        'primary_photo',
        'price_fbu',
        'price_usd',
        'price_eur',
        'description',
        'full_description',
        'size',
        'bed_type',
        'view',
        'amenities',
        'gallery',
        'status',
        'featured',
    ];

    protected $casts = [
        'amenities' => 'array',
        'gallery' => 'array',
        'price_fbu' => 'decimal:2',
        'price_usd' => 'decimal:2',
        'price_eur' => 'decimal:2',
        'status' => 'boolean',
        'featured' => 'boolean',
    ];

    public function bookings()
    {
        return $this->morphMany(Booking::class, 'bookable');
    }

    /**
     * Get all individual rooms for this accommodation category
     */
    public function rooms()
    {
        return $this->hasMany(Room::class);
    }

    /**
     * Get the pricing rule for this accommodation
     */
    public function pricingRule()
    {
        return $this->hasOne(PricingRule::class);
    }

    /**
     * Check if accommodation category has available rooms for given dates
     */
    public function isAvailable($checkIn, $checkOut)
    {
        // If we have individual rooms, check if any room is available
        if ($this->rooms()->exists()) {
            return $this->rooms()
                ->where('is_available', true)
                ->get()
                ->filter(function($room) use ($checkIn, $checkOut) {
                    return $room->isAvailableForDates($checkIn, $checkOut);
                })
                ->isNotEmpty();
        }

        // Fallback to old logic for backward compatibility
        $conflictingBookings = $this->bookings()
            ->where('status', 'confirmed')
            ->where(function($query) use ($checkIn, $checkOut) {
                $query->whereBetween('check_in', [$checkIn, $checkOut])
                    ->orWhereBetween('check_out', [$checkIn, $checkOut])
                    ->orWhere(function($q) use ($checkIn, $checkOut) {
                        $q->where('check_in', '<=', $checkIn)
                          ->where('check_out', '>=', $checkOut);
                    });
            })
            ->exists();

        return !$conflictingBookings;
    }

    /**
     * Get available rooms for given dates
     */
    public function getAvailableRooms($checkIn, $checkOut)
    {
        return $this->rooms()
            ->where('is_available', true)
            ->with('accommodation') // Eager load accommodation for display_name accessor
            ->get()
            ->filter(function($room) use ($checkIn, $checkOut) {
                return $room->isAvailableForDates($checkIn, $checkOut);
            })
            ->values();
    }
}
