<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Contact;
use App\Models\Booking;
use App\Models\Accommodation;
use App\Models\Room;
use App\Models\EventRoom;
use App\Models\Service;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Inertia\Inertia;
use Inertia\Response;

class DashboardController extends Controller
{
    public function fetchStats()
    {
        $now = Carbon::now();
        $startOfMonth = $now->copy()->startOfMonth();
        $endOfMonth = $now->copy()->endOfMonth();
        $lastMonth = $now->copy()->subMonth();
        
        // Basic counts
        $totalRoomCategories = Accommodation::where('status', 1)->count();
        $totalRoomsCount = Room::count();
        
        // Calculate actually available rooms (not booked today)
        $today = $now->format('Y-m-d');
        $bookedRoomIds = Booking::where('status', 'confirmed')
            ->whereNotNull('room_id')
            ->whereNotNull('check_in')
            ->whereNotNull('check_out')
            ->where('check_in', '<=', $today)
            ->where('check_out', '>', $today)
            ->pluck('room_id')
            ->unique()
            ->toArray();
        
        $availableRooms = Room::where('is_available', true)
            ->whereNotIn('id', $bookedRoomIds)
            ->count();
        
        $totalEventRooms = EventRoom::where('status', 1)->count();
        $totalServices = Service::where('status', true)->count();
        $unreadMessages = Contact::where('is_read', false)->count();
        
        // Booking statistics
        $totalBookings = Booking::count();
        $confirmedBookings = Booking::where('status', 'confirmed')->count();
        
        // Today's confirmed bookings (check-ins for today)
        $todayConfirmedBookings = Booking::where('status', 'confirmed')
            ->whereDate('check_in', $today)
            ->count();
        $pendingBookings = Booking::where('status', 'pending')->count();
        $cancelledBookings = Booking::whereIn('status', ['cancelled', 'canceled'])->count();
        
        // Monthly booking statistics
        $monthlyBookings = Booking::whereBetween('created_at', [$startOfMonth, $endOfMonth])->count();
        $monthlyConfirmed = Booking::where('status', 'confirmed')
            ->whereBetween('created_at', [$startOfMonth, $endOfMonth])
            ->count();
        $lastMonthBookings = Booking::whereBetween('created_at', [
            $lastMonth->copy()->startOfMonth(),
            $lastMonth->copy()->endOfMonth()
        ])->count();
        
        // Revenue statistics
        $totalRevenue = Booking::where('status', 'confirmed')->sum('price') ?? 0;
        $monthlyRevenue = Booking::where('status', 'confirmed')
            ->whereBetween('created_at', [$startOfMonth, $endOfMonth])
            ->sum('price') ?? 0;
        $lastMonthRevenue = Booking::where('status', 'confirmed')
            ->whereBetween('created_at', [
                $lastMonth->copy()->startOfMonth(),
                $lastMonth->copy()->endOfMonth()
            ])
            ->sum('price') ?? 0;
        $revenueGrowth = $lastMonthRevenue > 0 
            ? round((($monthlyRevenue - $lastMonthRevenue) / $lastMonthRevenue) * 100, 1)
            : ($monthlyRevenue > 0 ? 100 : 0);
        
        // Occupancy statistics
        $bookedRooms = Booking::where('status', 'confirmed')
            ->whereNotNull('room_id')
            ->where(function($query) use ($now) {
                $query->where('check_in', '<=', $now)
                      ->where('check_out', '>=', $now);
            })
            ->distinct('room_id')
            ->count('room_id');
        $occupancyRate = $totalRoomsCount > 0 
            ? round(($bookedRooms / $totalRoomsCount) * 100, 1)
            : 0;
        
        // Booking trends (last 6 months)
        $bookingTrends = [];
        for ($i = 5; $i >= 0; $i--) {
            $month = $now->copy()->subMonths($i);
            $monthStart = $month->copy()->startOfMonth();
            $monthEnd = $month->copy()->endOfMonth();
            
            $monthConfirmed = Booking::where('status', 'confirmed')
                ->whereBetween('created_at', [$monthStart, $monthEnd])
                ->count();
            
            $monthPending = Booking::where('status', 'pending')
                ->whereBetween('created_at', [$monthStart, $monthEnd])
                ->count();
            
            $bookingTrends[] = [
                'month' => $month->format('M Y'),
                'confirmed' => $monthConfirmed,
                'pending' => $monthPending,
            ];
        }
        
        // Recent bookings (last 10)
        $recentBookings = Booking::with(['room.accommodation', 'accommodation'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get()
            ->map(function ($booking) {
                $roomDisplay = null;
                if ($booking->room) {
                    $category = $booking->accommodation->room_category ?? $booking->accommodation->room_name ?? 'N/A';
                    $roomDisplay = "Room {$booking->room->room_number} – {$category}";
                }
                
                return [
                    'id' => $booking->id,
                    'booking_reference' => $booking->booking_reference,
                    'full_name' => $booking->full_name,
                    'room' => $roomDisplay,
                    'check_in' => $booking->check_in,
                    'check_out' => $booking->check_out,
                    'status' => $booking->status,
                    'price' => $booking->price,
                    'created_at' => $booking->created_at,
                ];
            });
        
        // Room category distribution
        $roomCategories = Accommodation::where('status', 1)
            ->withCount(['rooms' => function($query) {
                $query->where('is_available', true);
            }])
            ->get()
            ->map(function ($accommodation) {
                return [
                    'category' => $accommodation->room_category ?? $accommodation->room_name,
                    'available_rooms' => $accommodation->rooms_count,
                ];
            });

        // Bookings by category
        $bookingsByCategory = Booking::where('status', 'confirmed')
            ->with(['accommodation', 'room.accommodation'])
            ->get()
            ->groupBy(function ($booking) {
                if ($booking->room && $booking->room->accommodation) {
                    return $booking->room->accommodation->room_category ?? $booking->room->accommodation->room_name ?? 'Unknown';
                } elseif ($booking->accommodation) {
                    return $booking->accommodation->room_category ?? $booking->accommodation->room_name ?? 'Unknown';
                }
                return 'Unknown';
            })
            ->map(function ($bookings, $category) {
                return [
                    'category' => $category,
                    'count' => $bookings->count(),
                    'revenue' => $bookings->sum('price'),
                ];
            })
            ->values()
            ->sortByDesc('count');

        // Revenue by category
        $revenueByCategory = $bookingsByCategory->map(function ($item) {
            return [
                'category' => $item['category'],
                'revenue' => round($item['revenue'], 2),
            ];
        })->values();

        // Booking status distribution
        $statusDistribution = [
            ['status' => 'Confirmed', 'count' => $confirmedBookings, 'color' => '#28a745'],
            ['status' => 'Pending', 'count' => $pendingBookings, 'color' => '#ffc107'],
            ['status' => 'Cancelled', 'count' => $cancelledBookings, 'color' => '#dc3545'],
        ];
        
        return Inertia::render('Dashboard', [
            'stats' => [
                'totalRoomCategories' => $totalRoomCategories,
                'totalRooms' => $totalRoomsCount,
                'availableRooms' => $availableRooms,
                'totalEventRooms' => $totalEventRooms,
                'totalServices' => $totalServices,
                'unreadMessages' => $unreadMessages,
                'totalBookings' => $totalBookings,
                'confirmedBookings' => $confirmedBookings,
                'todayConfirmedBookings' => $todayConfirmedBookings,
                'pendingBookings' => $pendingBookings,
                'cancelledBookings' => $cancelledBookings,
                'monthlyBookings' => $monthlyBookings,
                'monthlyConfirmed' => $monthlyConfirmed,
                'lastMonthBookings' => $lastMonthBookings,
                'totalRevenue' => round($totalRevenue, 2),
                'monthlyRevenue' => round($monthlyRevenue, 2),
                'lastMonthRevenue' => round($lastMonthRevenue, 2),
                'revenueGrowth' => $revenueGrowth,
                'bookedRooms' => $bookedRooms,
                'occupancyRate' => $occupancyRate,
                'currentDate' => $now->format('Y-m-d'),
                'currentDateTime' => $now->format('Y-m-d H:i:s'),
            ],
            'bookingTrends' => $bookingTrends,
            'recentBookings' => $recentBookings,
            'roomCategories' => $roomCategories,
            'bookingsByCategory' => $bookingsByCategory,
            'revenueByCategory' => $revenueByCategory,
            'statusDistribution' => $statusDistribution,
        ]);
    }
}
