<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Room;
use App\Models\Accommodation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use Inertia\Inertia;
use App\Traits\DeleteModelTrait;

class RoomController extends Controller
{
    use DeleteModelTrait;

    /**
     * Get accommodations list (helper method)
     */
    private function getAccommodationsList()
    {
        return Accommodation::where('status', true)
            ->orderBy('room_category')
            ->orderBy('room_name')
            ->get()
            ->map(function ($accommodation) {
                return [
                    'id' => $accommodation->id,
                    'name' => $accommodation->room_category ?? $accommodation->room_name,
                    'room_category' => $accommodation->room_category,
                ];
            });
    }

    /**
     * Display a listing of all rooms.
     */
    public function index()
    {
        $today = now()->format('Y-m-d');
        
        // Get rooms that are currently booked (confirmed bookings that span today)
        $bookedRoomIds = \App\Models\Booking::where('status', 'confirmed')
            ->whereNotNull('room_id')
            ->whereNotNull('check_in')
            ->whereNotNull('check_out')
            ->where('check_in', '<=', $today)
            ->where('check_out', '>', $today)
            ->pluck('room_id')
            ->unique()
            ->toArray();

        $rooms = Room::with('accommodation')
            ->orderBy('accommodation_id')
            ->orderBy('room_number')
            ->get()
            ->map(function ($room) use ($bookedRoomIds) {
                // A room is available only if: is_available is true AND not currently booked
                $isCurrentlyBooked = in_array($room->id, $bookedRoomIds);
                $effectiveAvailability = $room->is_available && !$isCurrentlyBooked;
                
                return [
                    'id' => $room->id,
                    'room_number' => $room->room_number,
                    'accommodation_id' => $room->accommodation_id,
                    'accommodation_name' => $room->accommodation->room_category ?? $room->accommodation->room_name ?? 'N/A',
                    'is_available' => $effectiveAvailability,
                    'is_booked' => $isCurrentlyBooked,
                    'base_availability' => $room->is_available,
                    'created_at' => $room->created_at,
                    'updated_at' => $room->updated_at,
                ];
            });

        return Inertia::render('rooms/Room', [
            'rooms' => $rooms,
            'accommodations' => $this->getAccommodationsList(),
        ]);
    }

    /**
     * Show the form for creating a new room.
     */
    public function create()
    {
        return Inertia::render('rooms/create/Room', [
            'accommodations' => $this->getAccommodationsList(),
        ]);
    }

    /**
     * Store a newly created room.
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();

            $validatedData = $request->validate([
                'room_number' => 'required|string|max:255',
                'accommodation_id' => 'required|exists:accommodations,id',
                'is_available' => 'boolean',
            ]);

            // Check if room number already exists for this accommodation
            $existingRoom = Room::where('accommodation_id', $validatedData['accommodation_id'])
                ->where('room_number', $validatedData['room_number'])
                ->first();

            if ($existingRoom) {
                throw ValidationException::withMessages([
                    'room_number' => 'This room number already exists for the selected accommodation category.',
                ]);
            }

            $room = new Room();
            $room->room_number = $validatedData['room_number'];
            $room->accommodation_id = $validatedData['accommodation_id'];
            $room->is_available = $validatedData['is_available'] ?? true;
            $room->save();

            DB::commit();

            return redirect()->route('admin.rooms')->with('success', [
                'message' => 'Room added successfully',
                'check' => false
            ]);
        } catch (ValidationException $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', [
                'message' => 'An error occurred: ' . $e->getMessage(),
                'check' => false
            ]);
        }
    }

    /**
     * Show the form for editing the specified room.
     */
    public function edit($id)
    {
        $room = Room::with('accommodation')->findOrFail($id);
        
        return response()->json([
            'id' => $room->id,
            'room_number' => $room->room_number,
            'accommodation_id' => $room->accommodation_id,
            'accommodation_name' => $room->accommodation->room_category ?? $room->accommodation->room_name,
            'is_available' => $room->is_available,
            'accommodations' => $this->getAccommodationsList(), // Include accommodations for dropdown
        ]);
    }

    /**
     * Update the specified room.
     */
    public function update(Request $request, $id)
    {
        try {
            DB::beginTransaction();

            $room = Room::findOrFail($id);

            $validatedData = $request->validate([
                'room_number' => 'required|string|max:255',
                'accommodation_id' => 'required|exists:accommodations,id',
                'is_available' => 'boolean',
            ]);

            // Check if room number already exists for this accommodation (excluding current room)
            $existingRoom = Room::where('accommodation_id', $validatedData['accommodation_id'])
                ->where('room_number', $validatedData['room_number'])
                ->where('id', '!=', $id)
                ->first();

            if ($existingRoom) {
                throw ValidationException::withMessages([
                    'room_number' => 'This room number already exists for the selected accommodation category.',
                ]);
            }

            $room->room_number = $validatedData['room_number'];
            $room->accommodation_id = $validatedData['accommodation_id'];
            $room->is_available = $validatedData['is_available'] ?? $room->is_available;
            $room->save();

            DB::commit();

            return redirect()->route('admin.rooms')->with('success', [
                'message' => 'Room updated successfully',
                'check' => false
            ]);
        } catch (ValidationException $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', [
                'message' => 'An error occurred: ' . $e->getMessage(),
                'check' => false
            ]);
        }
    }

    /**
     * Remove the specified room.
     */
    public function destroy($id)
    {
        return $this->deleteModel($id, new Room());
    }
}
